﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// <inheritdoc />
    [Authorize]
    [Route("api/id-proof")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class IdProofController : BaseController
    {
        /// <summary>
        /// The idproof services.
        /// </summary>
        private readonly IIdProofService idProofService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public IdProofController(IIdProofService idProofService, IAuditLogService auditLogService)
        {
            this.idProofService = idProofService;
            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] IdProofModel model, [FromHeader] LocationHeader header)
        {
            model = (IdProofModel)EmptyFilter.Handler(model);
            var response = await this.idProofService.InsertAsync(model);
            
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.IdProof,
                        LogDate = DateTime.Now,
                        LogFrom = (short)model.LoginRoleId,
                        LogDescription = $"<b>{model.CreatedByName}<b> has added <b>Id Proof</b> of <strong>{model.IdProofName}</strong> successfully.",
                        LocationId =Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }
           
            return this.Success(response);
        }

        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] IdProofModel model,[FromHeader]LocationHeader header)
        {
            model = (IdProofModel)EmptyFilter.Handler(model);
            var response = await this.idProofService.UpdateAsync(model);
           
            
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.IdProof,
                        LogDate = DateTime.Now,
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),                        
                        LogDescription = $"{model.CreatedByName} has updated <b>Id Proof</b> of <strong>{model.IdProofName}</strong> successfully.",
                        LocationId =Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }
            
            
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] IdProofModel model)
        {
            model = (IdProofModel)EmptyFilter.Handler(model);
            var response = await this.idProofService.FetchAllAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-active")]
        public async Task<ActionResult> FetchActiveAsync([FromBody] IdProofModel model)
        {
            model = (IdProofModel)EmptyFilter.Handler(model);
            var response = await this.idProofService.FetchActiveAllAsync(model);
            return this.Success(response);
        }


        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] IdProofModel model,[FromHeader] LocationHeader header)
        {
            model = (IdProofModel)EmptyFilter.Handler(model);
            var response = await this.idProofService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.IdProof,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.ModifiedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} the IdProof <b> {model.IdProofName}</b>  successfully",
                LocationId =Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }

    }
}

